<?php
defined('ABSPATH') || exit;

class IranDargah_Settings_Loader {
    
    public static function get_all_gateways_data() {
        $gateways = [
            self::get_sep_gateway_data(),
            self::get_zarinpal_gateway_data(),
            self::get_aqayepardakht_gateway_data(),
            self::get_bitpay_gateway_data(),
            self::get_payping_gateway_data(),
            self::get_bahamta_gateway_data(),
            self::get_idpay_gateway_data(),
            self::get_irankish_gateway_data(),
            self::get_parsian_gateway_data(),
            self::get_pasargad_gateway_data(),
            self::get_rayanpay_gateway_data(),
            self::get_melli_gateway_data(),
            self::get_sepehrpay_gateway_data(),
            self::get_fanava_gateway_data(),
            self::get_mellat_gateway_data(),
            self::get_nextpay_gateway_data(),
            self::get_payment4_gateway_data(),
        ];

        return array_filter($gateways); // حذف درگاه‌هایی که null برگردانده‌اند
    }
    
    public static function get_sep_gateway_data() {
        $settings = get_option('woocommerce_WC_Sep_Payment_Gateway_settings', []);
        $accounts = get_option('woocommerce_sep_accounts', []);
        
        $callback_url = WC()->api_request_url('WC_Sep_Payment_Gateway');
        
        return [
            'id' => 'WC_Sep_Payment_Gateway',
            'title' => __('بانک سامان', 'irandargah'),
            'description' => __('درگاه پرداخت بانک سامان برای ووکامرس', 'irandargah'),
            'icon' => self::get_sep_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('بانک سامان', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'merchantcode' => [
                            'title' => __('کد پذیرنده (Terminal ID)', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['merchantcode'] ?? '',
                            'description' => __('کد پذیرنده دریافتی از بانک سامان', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل بانک سامان تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('پرداخت با موفقیت انجام شد.', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت ناموفق بود.', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }
    
    public static function get_zarinpal_gateway_data() {
        $settings = get_option('woocommerce_WC_ZPal_settings', []);
        
        $callback_url = WC()->api_request_url('WC_ZPal');
        
        return [
            'id' => 'WC_ZPal',
            'title' => __('زرین‌پال', 'irandargah'),
            'description' => __('درگاه پرداخت امن زرین‌پال', 'irandargah'),
            'icon' => self::get_zarinpal_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('پرداخت امن زرین‌پال', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'merchantcode' => [
                            'title' => __('مرچنت کد', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['merchantcode'] ?? '',
                            'description' => __('مرچنت کد دریافتی از زرین‌پال', 'irandargah'),
                            'placeholder' => __('مثال: 12345678-1234-1234-1234-123456789012', 'irandargah'),
                        ],
                        'access_token' => [
                            'title' => __('توکن دسترسی', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['access_token'] ?? '',
                            'description' => __('توکن دسترسی برای سرویس استرداد وجه (اختیاری)', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'fee_payer' => [
                            'title' => __('کسر کارمزد از', 'irandargah'),
                            'type' => 'select',
                            'value' => $settings['fee_payer'] ?? 'merchant',
                            'options' => [
                                'merchant' => __('پذیرنده', 'irandargah'),
                                'customer' => __('خریدار', 'irandargah'),
                            ],
                            'description' => __('انتخاب کنید کارمزد از چه کسی کسر شود', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل زرین‌پال تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_message' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_message'] ?? __('پرداخت با موفقیت انجام شد.', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_message' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_message'] ?? __('پرداخت ناموفق بود.', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                        'instructions' => [
                            'title' => __('دستورالعمل‌های پس از خرید', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['instructions'] ?? '',
                            'description' => __('دستورالعمل‌هایی که پس از پرداخت به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    /**
     * ساخت ساختار تنظیمات داینامیک برای درگاه‌های قدیمی
     *
     * @param string $gateway_id  شناسه درگاه در ووکامرس (id کلاس)
     * @param array  $meta        اطلاعات نمایشی (title, description)
     *
     * @return array
     */
    private static function get_dynamic_gateway_data($gateway_id, $meta) {
        $settings = get_option("woocommerce_{$gateway_id}_settings", []);

        $enabled = $settings['enabled'] ?? 'no';
        $sandbox = $settings['sandbox'] ?? 'no';

        $basic_fields = [];
        if (isset($settings['title'])) {
            $basic_fields['title'] = [
                'title' => __('عنوان درگاه', 'irandargah'),
                'type'  => 'text',
                'value' => $settings['title'],
                'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
            ];
        }

        if (isset($settings['description'])) {
            $basic_fields['description'] = [
                'title' => __('توضیحات', 'irandargah'),
                'type'  => 'textarea',
                'value' => $settings['description'],
                'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
            ];
        }

        // فیلدهای حساب/شناسه‌ها
        $account_fields = [];
        $message_fields = [];

        foreach ($settings as $key => $value) {
            if (in_array($key, ['enabled', 'sandbox', 'title', 'description'], true)) {
                continue;
            }

            $lower = strtolower($key);

            // فیلدهای پیام‌ها
            if (strpos($lower, 'message') !== false || strpos($lower, 'massage') !== false) {
                $message_fields[$key] = [
                    'title' => self::human_readable_key($key),
                    'type'  => 'textarea',
                    'value' => $value,
                    'description' => '',
                ];
                continue;
            }

            // فیلدهای حساب / اطلاعات اتصال
            if (preg_match('/merchant|terminal|pin|api|token|key|username|password|iban|account/i', $key)) {
                $type = (preg_match('/password|token|key/i', $key)) ? 'password' : 'text';

                $account_fields[$key] = [
                    'title' => self::human_readable_key($key),
                    'type'  => $type,
                    'value' => $value,
                    'description' => '',
                ];
                continue;
            }
        }

        // فیلد sandbox اگر وجود داشته باشد
        if (isset($settings['sandbox'])) {
            $account_fields = array_merge([
                'sandbox' => [
                    'title' => __('حالت آزمایشی', 'irandargah'),
                    'type'  => 'checkbox',
                    'value' => $settings['sandbox'],
                    'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                    'description' => __('در صورت نیاز به تست، این گزینه را فعال کنید.', 'irandargah'),
                ],
            ], $account_fields);
        }

        $sections = [];

        if (!empty($basic_fields)) {
            $sections['basic'] = [
                'title'  => __('تنظیمات پایه', 'irandargah'),
                'fields' => $basic_fields,
            ];
        }

        if (!empty($account_fields)) {
            $sections['account'] = [
                'title'  => __('تنظیمات حساب / اتصال', 'irandargah'),
                'fields' => $account_fields,
            ];
        }

        if (!empty($message_fields)) {
            $sections['messages'] = [
                'title'  => __('پیام‌ها', 'irandargah'),
                'fields' => $message_fields,
            ];
        }

        // اگر هیچ سکشنی ساخته نشد، یک سکشن خالی برای جلوگیری از خطا برمی‌گردانیم
        if (empty($sections)) {
            $sections['basic'] = [
                'title'  => __('تنظیمات', 'irandargah'),
                'fields' => [],
            ];
        }

        return [
            'id'                => $gateway_id,
            'title'             => $meta['title'],
            'description'       => $meta['description'],
            'icon'              => self::get_generic_icon_svg(),
            'enabled'           => $enabled,
            'sandbox'           => $sandbox,
            'settings_sections' => $sections,
        ];
    }

    /**
     * تبدیل کلید تنظیم به عنوان قابل‌خواندن
     */
    private static function human_readable_key($key) {
        $key = str_replace(['_', '-'], ' ', $key);
        $key = preg_replace('/\s+/', ' ', $key);
        return ucwords($key);
    }
    
    public static function save_gateway_settings($gateway_id, $settings) {
        $option_name = "woocommerce_{$gateway_id}_settings";
        $current_settings = get_option($option_name, []);
        
        foreach ($settings as $key => $value) {
            // فیلدهای بولی
            if ($key === 'enabled' || $key === 'sandbox' || in_array($key, ['ioserver', 'multi', 'is_finish_order'], true)) {
                $current_settings[$key] = $value === 'yes' ? 'yes' : 'no';
            } 
            // فیلدهای select
            elseif (in_array($key, ['fee_payer', 'order_status', 'vahed'], true)) {
                $current_settings[$key] = sanitize_text_field($value);
            } 
            // فیلدهای متنی (حساب/اتصال)
            elseif (in_array($key, [
                'title',
                'description',
                'merchantcode',
                'access_token',
                'pin',
                'api',
                'api_key',
                'paypingToken',
                'merchantid',
                'merchant_id',
                'merchant',
                'terminal_id',
                'terminalid',
                'terminal',
                'terminalID',
                'username',
                'password',
                'terminal_key',
                'pasargad_terminal_id',
                'pasargad_merchant_id',
                'pasargad_username',
                'pasargad_password',
                'base_url',
                'acceptorId',
                'pub_key',
                'discount_percent',
            ], true)) {
                $current_settings[$key] = sanitize_text_field($value);
            } 
            // فیلدهای متنی بلند (پیام‌ها)
            elseif (in_array($key, [
                'success_massage',
                'success_message',
                'failed_massage',
                'failed_message',
                'instructions',
                'succeeded_message',
                'canceled_message',
                'cancelled_massage',
                'connecting_message',
                'connection_error_massage',
            ], true)) {
                $current_settings[$key] = sanitize_textarea_field($value);
            } 
            // سایر فیلدها
            else {
                $current_settings[$key] = sanitize_text_field($value);
            }
        }
        
        update_option($option_name, $current_settings);
        
        return true;
    }
    
    public static function toggle_gateway($gateway_id, $enabled) {
        $option_name = "woocommerce_{$gateway_id}_settings";
        $settings = get_option($option_name, []);
        if (!is_array($settings)) {
            $settings = [];
        }
        $settings['enabled'] = $enabled ? 'yes' : 'no';
        update_option($option_name, $settings);

        // ثبت وضعیت نهایی درگاه‌ها برای کنترل مستقل از رابط ووکامرس
        $statuses = get_option('irandargah_gateway_statuses', []);
        if (!is_array($statuses)) {
            $statuses = [];
        }
        $statuses[$gateway_id] = $enabled ? 1 : 0;
        update_option('irandargah_gateway_statuses', $statuses);

        return true;
    }
    
    private static function get_sep_icon_svg() {
        return '<svg width="48" height="48" viewBox="0 0 48 48" fill="none" xmlns="http://www.w3.org/2000/svg">
            <rect width="48" height="48" rx="12" fill="#0066CC"/>
            <path d="M24 12C17.373 12 12 17.373 12 24s5.373 12 12 12 12-5.373 12-12S30.627 12 24 12zm0 20c-4.418 0-8-3.582-8-8s3.582-8 8-8 8 3.582 8 8-3.582 8-8 8zm-2-12v8l6 4 2-1.5-5-3V20h-3z" fill="white"/>
        </svg>';
    }
    
    private static function get_zarinpal_icon_svg() {
        return '<svg width="48" height="48" viewBox="0 0 48 48" fill="none" xmlns="http://www.w3.org/2000/svg">
            <rect width="48" height="48" rx="12" fill="#0A33FF"/>
            <path d="M24 8C15.163 8 8 15.163 8 24s7.163 16 16 16 16-7.163 16-16S32.837 8 24 8zm0 28c-6.627 0-12-5.373-12-12S17.373 12 24 12s12 5.373 12 12-5.373 12-12 12zm-4-16v8l7 4.5L30 28l-6-3.75V16h-4z" fill="white"/>
        </svg>';
    }

    /**
     * آیکن عمومی برای درگاه‌هایی که آیکن اختصاصی در داشبورد ندارند
     */
    private static function get_generic_icon_svg() {
        return '<svg width="48" height="48" viewBox="0 0 48 48" fill="none" xmlns="http://www.w3.org/2000/svg">
            <rect width="48" height="48" rx="12" fill="#4A5568"/>
            <path d="M16 14h16c1.105 0 2 .895 2 2v16c0 1.105-.895 2-2 2H16c-1.105 0-2-.895-2-2V16c0-1.105.895-2 2-2zm2 4v2h12v-2H18zm0 6v2h8v-2h-8z" fill="white"/>
        </svg>';
    }

    // ========== توابع اختصاصی برای هر درگاه ==========

    public static function get_aqayepardakht_gateway_data() {
        $settings = get_option('woocommerce_WC_aqayepardakht_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_aqayepardakht');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'WC_aqayepardakht',
            'title' => __('آقای پرداخت', 'irandargah'),
            'description' => __('درگاه پرداخت آقای پرداخت برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('آقای پرداخت', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'pin' => [
                            'title' => __('پین', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['pin'] ?? '',
                            'description' => __('پین درگاه آقای پرداخت', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل آقای پرداخت تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_bitpay_gateway_data() {
        $settings = get_option('woocommerce_WC_bitpayir2_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_bitpayir2');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'WC_bitpayir2',
            'title' => __('بیت‌پی', 'irandargah'),
            'description' => __('درگاه پرداخت بیت‌پی برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('پرداخت امن بیت پی', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'api' => [
                            'title' => __('API', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['api'] ?? '',
                            'description' => __('API درگاه بیت پی', 'irandargah'),
                            'placeholder' => __('مثال: adxcv-zzadq-polkjsad-...', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل بیت پی تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                        'cancelled_massage' => [
                            'title' => __('پیام انصراف از پرداخت', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['cancelled_massage'] ?? __('پرداخت به دلیل انصراف شما ناتمام باقی ماند .', 'irandargah'),
                            'description' => __('پیام نمایش داده شده هنگام انصراف کاربر', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_payping_gateway_data() {
        $settings = get_option('woocommerce_WC_payping_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_payping');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'WC_payping',
            'title' => __('پی‌پینگ', 'irandargah'),
            'description' => __('درگاه پرداخت پی‌پینگ برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('پرداخت از طریق پی‌پینگ', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'paypingToken' => [
                            'title' => __('توکن', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['paypingToken'] ?? '',
                            'description' => __('توکن درگاه پی‌پینگ', 'irandargah'),
                            'placeholder' => __('مثال: xxxxx-xxxxx-xxxxx', 'irandargah'),
                        ],
                        'ioserver' => [
                            'title' => __('سرور خارج', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['ioserver'] ?? 'no',
                            'label' => __('اتصال به سرور خارج', 'irandargah'),
                            'description' => __('در صورت تیک خوردن، درگاه به سرور خارج از کشور متصل می‌شود', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل پی‌پینگ تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_bahamta_gateway_data() {
        $settings = get_option('woocommerce_bahamta_payment_gateway_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('Bahamta_Payment_Gateway');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'bahamta_payment_gateway',
            'title' => __('باهمتا', 'irandargah'),
            'description' => __('درگاه پرداخت باهمتا برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('باهمتا', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'api_key' => [
                            'title' => __('کلید درگاه پرداخت', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['api_key'] ?? '',
                            'description' => __('کلید اتصال به درگاه پرداخت', 'irandargah'),
                            'placeholder' => __('مثال: xxxxx-xxxxx-xxxxx', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل باهمتا تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'succeeded_message' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['succeeded_message'] ?? __('پرداخت با موفقیت انجام شد.', 'irandargah'),
                            'description' => __('پیام نمایش داده شده پس از پرداخت موفق', 'irandargah'),
                        ],
                        'failed_message' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_message'] ?? __('پرداخت با خطا روبرو شد.', 'irandargah'),
                            'description' => __('پیام نمایش داده شده پس از پرداخت ناموفق', 'irandargah'),
                        ],
                        'canceled_message' => [
                            'title' => __('پیام انصراف از پرداخت', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['canceled_message'] ?? __('پرداخت از طرف خریدار لغو شد.', 'irandargah'),
                            'description' => __('پیام نمایش داده شده هنگام انصراف کاربر', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_idpay_gateway_data() {
        $settings = get_option('woocommerce_WC_IDPay_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('wc_idpay');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'WC_IDPay',
            'title' => __('IDPay', 'irandargah'),
            'description' => __('درگاه پرداخت IDPay برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('IDPay payment gateway', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'api_key' => [
                            'title' => __('API Key', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['api_key'] ?? '',
                            'description' => __('API Key دریافتی از IDPay', 'irandargah'),
                            'placeholder' => __('مثال: xxxxx-xxxxx-xxxxx', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'order_status' => [
                            'title' => __('وضعیت سفارش پس از پرداخت', 'irandargah'),
                            'type' => 'select',
                            'value' => $settings['order_status'] ?? 'completed',
                            'options' => [
                                'completed' => __('تکمیل شده', 'irandargah'),
                                'processing' => __('در حال پردازش', 'irandargah'),
                            ],
                            'description' => __('وضعیت سفارش پس از پرداخت موفق', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل IDPay تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_message' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_message'] ?? __('Your payment has been successfully completed. Track id: {track_id}', 'irandargah'),
                            'description' => __('از {track_id} و {order_id} برای نمایش کد رهگیری و شماره سفارش استفاده کنید', 'irandargah'),
                        ],
                        'failed_message' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_message'] ?? __('Your payment has failed. Please try again or contact the site administrator in case of a problem.', 'irandargah'),
                            'description' => __('از {track_id} و {order_id} برای نمایش کد رهگیری و شماره سفارش استفاده کنید', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_irankish_gateway_data() {
        $settings = get_option('woocommerce_irankish_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_Gateway_Irankish');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'irankish',
            'title' => __('ایران‌کیش', 'irandargah'),
            'description' => __('درگاه پرداخت ایران‌کیش برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('ایران‌کیش', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'terminalID' => [
                            'title' => __('شماره پایانه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['terminalID'] ?? '',
                            'description' => __('شماره پایانه درگاه ایران‌کیش', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'password' => [
                            'title' => __('کلمه عبور', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['password'] ?? '',
                            'description' => __('کلمه عبور درگاه ایران‌کیش', 'irandargah'),
                        ],
                        'acceptorId' => [
                            'title' => __('شماره پذیرنده', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['acceptorId'] ?? '',
                            'description' => __('شماره پذیرنده درگاه ایران‌کیش', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'pub_key' => [
                            'title' => __('کلید عمومی', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['pub_key'] ?? '',
                            'description' => __('کلید عمومی درگاه ایران‌کیش', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل ایران‌کیش تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {refrenceID} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                        'cancelled_massage' => [
                            'title' => __('پیام انصراف از پرداخت', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['cancelled_massage'] ?? __('پرداخت به دلیل انصراف شما ناتمام باقی ماند .', 'irandargah'),
                            'description' => __('پیام نمایش داده شده هنگام انصراف کاربر', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_parsian_gateway_data() {
        $settings = get_option('woocommerce_bankparsian_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_Gateway_Bankparsian');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'bankparsian',
            'title' => __('بانک پارسیان', 'irandargah'),
            'description' => __('درگاه پرداخت بانک پارسیان برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('بانک پارسیان', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'pin' => [
                            'title' => __('شناسه پذیرنده', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['pin'] ?? '',
                            'description' => __('پین کد دریافت شده از بانک پارسیان', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل بانک پارسیان تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'connecting_message' => [
                            'title' => __('پیام اتصال به بانک', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['connecting_message'] ?? __('در حال اتصال به بانک...', 'irandargah'),
                            'description' => __('پیام نمایش داده شده هنگام اتصال به بانک', 'irandargah'),
                        ],
                        'connection_error_massage' => [
                            'title' => __('پیام خطا در اتصال به بانک', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['connection_error_massage'] ?? __('خطا در اتصال به بانک، لطفا مجددا تلاش نمایید.', 'irandargah'),
                            'description' => __('پیام نمایش داده شده پس از خطا در اتصال', 'irandargah'),
                        ],
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما، پرداخت سفارش شما با موفقیت انجام شد، کد رهگیری خود را یادداشت نمایید.', 'irandargah'),
                            'description' => __('پیام نمایش داده شده پس از پرداخت موفق', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت سفارش شما ناموفق بوده است. لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید.', 'irandargah'),
                            'description' => __('پیام نمایش داده شده پس از پرداخت ناموفق', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_pasargad_gateway_data() {
        $settings = get_option('woocommerce_pasargad_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_Pasargad_Gateway');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'pasargad',
            'title' => __('بانک پاسارگاد', 'irandargah'),
            'description' => __('درگاه پرداخت بانک پاسارگاد برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('بانک پاسارگاد', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'pasargad_terminal_id' => [
                            'title' => __('شماره ترمینال', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['pasargad_terminal_id'] ?? '',
                            'description' => __('شماره ترمینال درگاه پاسارگاد', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'pasargad_merchant_id' => [
                            'title' => __('شماره فروشگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['pasargad_merchant_id'] ?? '',
                            'description' => __('شماره فروشگاه درگاه پاسارگاد', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'pasargad_username' => [
                            'title' => __('نام کاربری', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['pasargad_username'] ?? '',
                            'description' => __('نام کاربری درگاه پاسارگاد', 'irandargah'),
                        ],
                        'pasargad_password' => [
                            'title' => __('کلمه عبور', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['pasargad_password'] ?? '',
                            'description' => __('کلمه عبور درگاه پاسارگاد', 'irandargah'),
                        ],
                        'base_url' => [
                            'title' => __('آدرس پایه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['base_url'] ?? 'https://pep.shaparak.ir/pepg',
                            'description' => __('آدرس پایه API پاسارگاد', 'irandargah'),
                        ],
                        'is_finish_order' => [
                            'title' => __('وضعیت سفارش پس از پرداخت', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['is_finish_order'] ?? 'no',
                            'label' => __('سفارش به وضعیت تکمیل تغییر یابد', 'irandargah'),
                            'description' => __('در صورت فعال بودن، سفارش به وضعیت تکمیل تغییر می‌یابد، در غیر این صورت به وضعیت در حال آماده‌سازی', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل پاسارگاد تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} و {SaleOrderId} برای نمایش کد رهگیری و شماره درخواست استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                        'cancelled_massage' => [
                            'title' => __('پیام انصراف از پرداخت', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['cancelled_massage'] ?? __('پرداخت به دلیل انصراف شما ناتمام باقی ماند .', 'irandargah'),
                            'description' => __('پیام نمایش داده شده هنگام انصراف کاربر', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_rayanpay_gateway_data() {
        $settings = get_option('woocommerce_rayanpay_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_RayanPay');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'rayanpay',
            'title' => __('رایان‌پی', 'irandargah'),
            'description' => __('درگاه پرداخت رایان‌پی برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('RayanPay Payment Gateway', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'merchantid' => [
                            'title' => __('RayanPay Merchant ID', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['merchantid'] ?? '',
                            'description' => __('شناسه مرچنت رایان‌پی', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل رایان‌پی تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_melli_gateway_data() {
        $settings = get_option('woocommerce_melli_pay_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_Gateway_Melli');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'melli_pay',
            'title' => __('بانک ملّی', 'irandargah'),
            'description' => __('درگاه پرداخت بانک ملّی برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('بانک ملی', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'merchant_id' => [
                            'title' => __('شماره پذیرنده', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['merchant_id'] ?? '',
                            'description' => __('شماره پذیرنده درگاه بانک ملی', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'terminal_id' => [
                            'title' => __('شماره ترمینال', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['terminal_id'] ?? '',
                            'description' => __('شماره ترمینال درگاه بانک ملی', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'terminal_key' => [
                            'title' => __('کلید تراکنش', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['terminal_key'] ?? '',
                            'description' => __('کلید تراکنش درگاه بانک ملی', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل بانک ملی تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما، سفارش شما با موفقیت و با کد رهگیری {transaction_id} و شناسه تراکنش  {SaleOrderId} پرداخت شد. ', 'irandargah'),
                            'description' => __('از {transaction_id} و {SaleOrderId} برای نمایش کد رهگیری و شناسه تراکنش استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                        'cancelled_massage' => [
                            'title' => __('پیام انصراف از پرداخت', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['cancelled_massage'] ?? __('پرداخت به دلیل انصراف شما ناتمام باقی ماند .', 'irandargah'),
                            'description' => __('پیام نمایش داده شده هنگام انصراف کاربر', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_sepehrpay_gateway_data() {
        $settings = get_option('woocommerce_pas_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_pas');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'pas',
            'title' => __('سپهر (SepehrPay)', 'irandargah'),
            'description' => __('درگاه پرداخت سپهر برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('پرداخت الکترونیک سپهر', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'terminal' => [
                            'title' => __('ترمینال', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['terminal'] ?? '',
                            'description' => __('ترمینال درگاه پرداخت الکترونیک سپهر', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'vahed' => [
                            'title' => __('واحد پولی', 'irandargah'),
                            'type' => 'select',
                            'value' => $settings['vahed'] ?? 'rial',
                            'options' => [
                                'rial' => __('ریال', 'irandargah'),
                                'toman' => __('تومان', 'irandargah'),
                            ],
                            'description' => __('تعیین کنید واحد پول وب‌سایت شما ریال یا تومان است', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل سپهر تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_fanava_gateway_data() {
        $settings = get_option('woocommerce_Fanava_settings', []);
        
        return [
            'id' => 'Fanava',
            'title' => __('فن‌آوا کارت', 'irandargah'),
            'description' => __('درگاه پرداخت فن‌آوا کارت برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('فن آوا کارت', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'merchant' => [
                            'title' => __('کد پذیرنده (مرچنت)', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['merchant'] ?? '',
                            'description' => __('کد پذیرنده درگاه فن آوا کارت', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'username' => [
                            'title' => __('نام کاربری', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['username'] ?? '',
                            'description' => __('نام کاربری درگاه فن آوا کارت', 'irandargah'),
                        ],
                        'password' => [
                            'title' => __('کلمه عبور', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['password'] ?? '',
                            'description' => __('کلمه عبور درگاه فن آوا کارت', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_mellat_gateway_data() {
        $settings = get_option('woocommerce_WC_BankMellat_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_BankMellat');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'WC_BankMellat',
            'title' => __('بانک ملت', 'irandargah'),
            'description' => __('درگاه پرداخت بانک ملت برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('بانک ملت', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'terminal' => [
                            'title' => __('ترمینال آیدی', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['terminal'] ?? '',
                            'description' => __('شماره ترمینال درگاه بانک ملت', 'irandargah'),
                            'placeholder' => __('مثال: 123456789', 'irandargah'),
                        ],
                        'username' => [
                            'title' => __('نام کاربری', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['username'] ?? '',
                            'description' => __('نام کاربری درگاه بانک ملت', 'irandargah'),
                        ],
                        'password' => [
                            'title' => __('کلمه عبور', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['password'] ?? '',
                            'description' => __('کلمه عبور درگاه بانک ملت', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل بانک ملت تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} و {SaleOrderId} برای نمایش کد رهگیری و شماره درخواست استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                        'cancelled_massage' => [
                            'title' => __('پیام انصراف از پرداخت', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['cancelled_massage'] ?? __('پرداخت به دلیل انصراف شما ناتمام باقی ماند .', 'irandargah'),
                            'description' => __('پیام نمایش داده شده هنگام انصراف کاربر', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_nextpay_gateway_data() {
        $settings = get_option('woocommerce_WC_NextPay_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_NextPay');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'WC_NextPay',
            'title' => __('نکست‌پی', 'irandargah'),
            'description' => __('درگاه پرداخت نکست‌پی برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('پرداخت امن نکست پی', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'merchantcode' => [
                            'title' => __('کلید مجوزدهی', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['merchantcode'] ?? '',
                            'description' => __('کلید API نکست پی', 'irandargah'),
                            'placeholder' => __('مثال: xxxxx-xxxxx-xxxxx', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل نکست پی تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
                'messages' => [
                    'title' => __('پیام‌ها', 'irandargah'),
                    'fields' => [
                        'success_massage' => [
                            'title' => __('پیام پرداخت موفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['success_massage'] ?? __('با تشکر از شما . سفارش شما با موفقیت پرداخت شد .', 'irandargah'),
                            'description' => __('از {transaction_id} برای نمایش کد رهگیری استفاده کنید', 'irandargah'),
                        ],
                        'failed_massage' => [
                            'title' => __('پیام پرداخت ناموفق', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['failed_massage'] ?? __('پرداخت شما ناموفق بوده است . لطفا مجددا تلاش نمایید یا در صورت بروز اشکال با مدیر سایت تماس بگیرید .', 'irandargah'),
                            'description' => __('از {fault} برای نمایش دلیل خطا استفاده کنید', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }

    public static function get_payment4_gateway_data() {
        $settings = get_option('woocommerce_WC_Payment4_settings', []);
        $callback_url = '';
        if (function_exists('WC') && WC()) {
            try {
                $callback_url = WC()->api_request_url('WC_Payment4_callback');
            } catch (Exception $e) {
                $callback_url = '';
            }
        }
        
        return [
            'id' => 'WC_Payment4',
            'title' => __('پیمنت۴', 'irandargah'),
            'description' => __('درگاه پرداخت Payment4 برای ووکامرس', 'irandargah'),
            'icon' => self::get_generic_icon_svg(),
            'enabled' => $settings['enabled'] ?? 'no',
            'sandbox' => $settings['sandbox'] ?? 'no',
            'settings_sections' => [
                'basic' => [
                    'title' => __('تنظیمات پایه', 'irandargah'),
                    'fields' => [
                        'title' => [
                            'title' => __('عنوان درگاه', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['title'] ?? __('Payment4', 'irandargah'),
                            'description' => __('عنوانی که در صفحه پرداخت نمایش داده می‌شود', 'irandargah'),
                        ],
                        'description' => [
                            'title' => __('توضیحات', 'irandargah'),
                            'type' => 'textarea',
                            'value' => $settings['description'] ?? '',
                            'description' => __('توضیحاتی که به مشتری نمایش داده می‌شود', 'irandargah'),
                        ],
                    ],
                ],
                'account' => [
                    'title' => __('تنظیمات حساب', 'irandargah'),
                    'fields' => [
                        'api_key' => [
                            'title' => __('API Key', 'irandargah'),
                            'type' => 'password',
                            'value' => $settings['api_key'] ?? '',
                            'description' => __('API Key دریافتی از Payment4', 'irandargah'),
                            'placeholder' => __('مثال: xxxxx-xxxxx-xxxxx', 'irandargah'),
                        ],
                        'sandbox' => [
                            'title' => __('حالت آزمایشی', 'irandargah'),
                            'type' => 'checkbox',
                            'value' => $settings['sandbox'] ?? 'no',
                            'label' => __('فعال‌سازی حالت Sandbox', 'irandargah'),
                            'description' => __('برای تست درگاه از حالت آزمایشی استفاده کنید', 'irandargah'),
                        ],
                        'discount_percent' => [
                            'title' => __('درصد تخفیف', 'irandargah'),
                            'type' => 'text',
                            'value' => $settings['discount_percent'] ?? '0',
                            'description' => __('درصد تخفیف برای پرداخت با Payment4 (اختیاری)', 'irandargah'),
                            'placeholder' => __('مثال: 5', 'irandargah'),
                        ],
                        'callback_url' => [
                            'title' => __('آدرس بازگشت (Callback URL)', 'irandargah'),
                            'type' => 'readonly',
                            'value' => $callback_url,
                            'description' => __('این آدرس را در پنل Payment4 تنظیم کنید', 'irandargah'),
                        ],
                    ],
                ],
            ],
        ];
    }
}


