<?php
defined('ABSPATH') || exit;

/**
 * کلاس برای پاک‌سازی اطلاعات افزونه‌های درگاه‌ها
 * این کلاس تمام اطلاعات سازنده، نسخه، آپدیت و نوتیفیکیشن‌ها را حذف می‌کند
 */
class IranDargah_Gateway_Cleanup {
    
    private static $gateway_plugins = [];
    
    public static function init() {
        // لیست تمام درگاه‌های ایرانی
        self::$gateway_plugins = [
            'aqayepardakht',
            'bahamta',
            'bitpay',
            'FanAva',
            'idpay',
            'irankish',
            'melli',
            'parsian',
            'pasargad',
            'payping',
            'rayanpay',
            'sepehrpay',
            'mellat',
            'nextpay',
            'payment4',
        ];
        
        // حذف اطلاعات سازنده
        add_filter('all_plugins', [__CLASS__, 'remove_plugin_author_info'], 999);
        add_filter('plugin_row_meta', [__CLASS__, 'remove_plugin_meta_links'], 999, 2);
        
        // حذف اطلاعات نسخه
        add_filter('all_plugins', [__CLASS__, 'remove_plugin_version_info'], 999);
        
        // غیرفعال کردن سیستم آپدیت
        add_filter('site_transient_update_plugins', [__CLASS__, 'disable_plugin_updates'], 999);
        add_filter('transient_update_plugins', [__CLASS__, 'disable_plugin_updates'], 999);
        add_filter('pre_set_site_transient_update_plugins', [__CLASS__, 'disable_plugin_updates'], 999);
        remove_action('load-update-core.php', 'wp_update_plugins');
        remove_action('load-update.php', 'wp_update_plugins');
        remove_action('load-update-core.php', 'wp_update_plugins');
        remove_action('admin_init', '_maybe_update_plugins');
        remove_action('wp_update_plugins', 'wp_update_plugins');
        wp_clear_scheduled_hook('wp_update_plugins');
        
        // حذف نوتیفیکیشن‌ها و هشدارها
        add_action('admin_init', [__CLASS__, 'remove_update_nags'], 999);
        add_action('admin_notices', [__CLASS__, 'remove_admin_notices'], 1);
        add_action('network_admin_notices', [__CLASS__, 'remove_admin_notices'], 1);
        add_filter('admin_footer_text', [__CLASS__, 'remove_footer_text'], 999);
        
        // حذف لینک‌های اکشن غیرضروری
        add_filter('plugin_action_links', [__CLASS__, 'remove_plugin_action_links'], 999, 2);
        add_filter('network_admin_plugin_action_links', [__CLASS__, 'remove_plugin_action_links'], 999, 2);
        
        // حذف ستون‌های غیرضروری از لیست افزونه‌ها
        add_filter('manage_plugins_columns', [__CLASS__, 'remove_plugin_columns'], 999);
        add_filter('manage_plugins-network_columns', [__CLASS__, 'remove_plugin_columns'], 999);
        
        // جلوگیری از نمایش اطلاعات در hover
        add_action('admin_head', [__CLASS__, 'hide_plugin_tooltips'], 999);
        add_action('admin_footer', [__CLASS__, 'hide_plugin_tooltips_js'], 999);
        
        // جلوگیری از چک آپدیت از API
        add_filter('http_request_args', [__CLASS__, 'disable_update_checks'], 5, 2);
        add_filter('plugins_api', [__CLASS__, 'disable_plugin_api'], 10, 3);
        add_filter('pre_http_request', [__CLASS__, 'short_circuit_update_requests'], 10, 3);
    }
    
    /**
     * حذف اطلاعات سازنده از لیست افزونه‌ها
     */
    public static function remove_plugin_author_info($plugins) {
        foreach (self::$gateway_plugins as $gateway) {
            $plugin_file = self::get_plugin_file($gateway);
            if ($plugin_file && isset($plugins[$plugin_file])) {
                // حذف نام سازنده
                if (isset($plugins[$plugin_file]['Author'])) {
                    $plugins[$plugin_file]['Author'] = '';
                }
                
                // حذف لینک سایت سازنده
                if (isset($plugins[$plugin_file]['AuthorURI'])) {
                    $plugins[$plugin_file]['AuthorURI'] = '';
                }
                
                // حذف توضیح افزونه
                if (isset($plugins[$plugin_file]['Description'])) {
                    $plugins[$plugin_file]['Description'] = __('درگاه پرداخت ایرانی', 'irandargah');
                }
                
                // حذف Plugin URI
                if (isset($plugins[$plugin_file]['PluginURI'])) {
                    $plugins[$plugin_file]['PluginURI'] = '';
                }
            }
        }
        
        return $plugins;
    }
    
    /**
     * حذف لینک‌های Plugin Meta
     */
    public static function remove_plugin_meta_links($links, $file) {
        foreach (self::$gateway_plugins as $gateway) {
            $plugin_file = self::get_plugin_file($gateway);
            if ($plugin_file === $file) {
                // حذف تمام لینک‌ها به جز لینک‌های ضروری
                $links = [];
            }
        }
        
        return $links;
    }
    
    /**
     * حذف اطلاعات نسخه
     */
    public static function remove_plugin_version_info($plugins) {
        foreach (self::$gateway_plugins as $gateway) {
            $plugin_file = self::get_plugin_file($gateway);
            if ($plugin_file && isset($plugins[$plugin_file])) {
                // حذف شماره نسخه
                if (isset($plugins[$plugin_file]['Version'])) {
                    $plugins[$plugin_file]['Version'] = '';
                }
            }
        }
        
        return $plugins;
    }
    
    /**
     * غیرفعال کردن سیستم آپدیت
     */
    public static function disable_plugin_updates($value) {
        if (!is_object($value) || !isset($value->response)) {
            return $value;
        }
        
        foreach (self::$gateway_plugins as $gateway) {
            $plugin_file = self::get_plugin_file($gateway);
            if ($plugin_file && isset($value->response[$plugin_file])) {
                unset($value->response[$plugin_file]);
            }
        }
        
        return $value;
    }
    
    /**
     * حذف Update Nag و نوتیفیکیشن‌ها
     */
    public static function remove_update_nags() {
        // حذف Update Nag
        remove_action('admin_notices', 'update_nag', 3);
        remove_action('network_admin_notices', 'update_nag', 3);
        
        // حذف Update Core Nag
        remove_action('admin_notices', 'maintenance_nag');
        
        // حذف Update Plugins Nag
        add_filter('pre_site_transient_update_plugins', '__return_null');
    }
    
    /**
     * حذف Admin Notices
     */
    public static function remove_admin_notices() {
        global $wp_filter;
        
        // حذف تمام admin_notices مربوط به بروزرسانی
        if (isset($wp_filter['admin_notices'])) {
            foreach ($wp_filter['admin_notices']->callbacks as $priority => $callbacks) {
                foreach ($callbacks as $callback_id => $callback) {
                    if (is_array($callback['function']) && is_object($callback['function'][0])) {
                        $class_name = get_class($callback['function'][0]);
                        if (strpos($class_name, 'Update') !== false || 
                            strpos($class_name, 'Plugin') !== false) {
                            unset($wp_filter['admin_notices']->callbacks[$priority][$callback_id]);
                        }
                    }
                }
            }
        }
        
        // حذف Warningهای افزونه‌های بدون منبع
        foreach (self::$gateway_plugins as $gateway) {
            $plugin_file = self::get_plugin_file($gateway);
            if ($plugin_file) {
                remove_action('after_plugin_row_' . $plugin_file, 'wp_plugin_update_row', 10);
            }
        }
    }
    
    /**
     * حذف لینک‌های اکشن غیرضروری
     */
    public static function remove_plugin_action_links($links, $file) {
        foreach (self::$gateway_plugins as $gateway) {
            $plugin_file = self::get_plugin_file($gateway);
            if ($plugin_file === $file) {
                // حذف تمام لینک‌ها به جز فعال/غیرفعال
                $new_links = [];
                foreach ($links as $key => $link) {
                    // فقط لینک فعال/غیرفعال را نگه دار
                    if (strpos($link, 'deactivate') !== false || strpos($link, 'activate') !== false) {
                        $new_links[$key] = $link;
                    }
                }
                $links = $new_links;
            }
        }
        
        return $links;
    }
    
    /**
     * حذف ستون‌های غیرضروری
     */
    public static function remove_plugin_columns($columns) {
        // حذف ستون "نویسنده"
        if (isset($columns['author'])) {
            unset($columns['author']);
        }
        
        // حذف ستون Version
        if (isset($columns['version'])) {
            unset($columns['version']);
        }
        
        return $columns;
    }
    
    /**
     * مخفی کردن Tooltip و Hover
     */
    public static function hide_plugin_tooltips() {
        $selectors = [];
        foreach (self::$gateway_plugins as $gateway) {
            $plugin_file = self::get_plugin_file($gateway);
            if ($plugin_file) {
                $plugin_slug = str_replace('/', '-', dirname($plugin_file));
                $selectors[] = 'tr[data-plugin="' . esc_attr($plugin_file) . '"]';
            }
        }
        
        if (!empty($selectors)) {
            echo '<style>
                ' . implode(', ', $selectors) . ' {
                    pointer-events: none;
                }
                ' . implode(', ', $selectors) . ' .row-actions {
                    display: none !important;
                }
                ' . implode(', ', $selectors) . ' .plugin-title strong::after {
                    display: none !important;
                }
                ' . implode(', ', $selectors) . ' .plugin-description {
                    display: none !important;
                }
                ' . implode(', ', $selectors) . ' .column-author,
                ' . implode(', ', $selectors) . ' .column-version {
                    display: none !important;
                }
            </style>';
        }
    }
    
    /**
     * دریافت مسیر فایل افزونه بر اساس نام درگاه
     */
    private static function get_plugin_file($gateway) {
        $plugin_base = 'irandargah/' . $gateway . '/';
        
        // بررسی فایل‌های مختلف برای هر درگاه
        $possible_files = [
            $plugin_base . 'index.php',
            $plugin_base . $gateway . '-gateway-wooc.php',
            $plugin_base . 'woo-' . $gateway . '-gateway.php',
            $plugin_base . 'class-wc-gateway-' . $gateway . '.php',
            $plugin_base . 'woocommerce-' . $gateway . '.php',
            $plugin_base . 'wszb-' . $gateway . '.php',
        ];
        
        // فایل‌های خاص برای درگاه‌های خاص
        $specific_files = [
            'aqayepardakht' => $plugin_base . 'index.php',
            'bahamta' => $plugin_base . 'bahamta-gateway-wooc.php',
            'bitpay' => $plugin_base . 'index.php',
            'FanAva' => $plugin_base . 'index.php',
            'idpay' => $plugin_base . 'woo-idpay-gateway.php',
            'irankish' => $plugin_base . 'class-wc-gateway-irankish.php',
            'melli' => $plugin_base . 'index.php',
            'parsian' => $plugin_base . 'parsian-woocommerce.php',
            'pasargad' => $plugin_base . 'woocommerce-pasargad.php',
            'payping' => $plugin_base . 'index.php',
            'rayanpay' => $plugin_base . 'rayanpay.php',
            'sepehrpay' => $plugin_base . 'wszb-sepehrpay.php',
            'mellat' => $plugin_base . 'index.php',
            'nextpay' => $plugin_base . 'index.php',
            'payment4' => $plugin_base . 'index.php',
        ];
        
        if (isset($specific_files[$gateway])) {
            $plugin_file = $specific_files[$gateway];
        } else {
            $plugin_file = $possible_files[0];
        }
        
        // بررسی وجود فایل
        $full_path = WP_PLUGIN_DIR . '/' . $plugin_file;
        if (file_exists($full_path)) {
            return $plugin_file;
        }
        
        return null;
    }
    
    /**
     * حذف Footer Text
     */
    public static function remove_footer_text($text) {
        return '';
    }
    
    /**
     * مخفی کردن Tooltip با JavaScript
     */
    public static function hide_plugin_tooltips_js() {
        $selectors = [];
        foreach (self::$gateway_plugins as $gateway) {
            $plugin_file = self::get_plugin_file($gateway);
            if ($plugin_file) {
                $selectors[] = 'tr[data-plugin="' . esc_js($plugin_file) . '"]';
            }
        }
        
        if (!empty($selectors)) {
            echo '<script>
                jQuery(document).ready(function($) {
                    var selectors = ' . json_encode($selectors) . ';
                    selectors.forEach(function(selector) {
                        $(selector).find(".row-actions").remove();
                        $(selector).find(".plugin-title strong").next().remove();
                        $(selector).find(".plugin-description").remove();
                        $(selector).find("td.column-author, td.column-version").remove();
                        $(selector).off("mouseenter mouseleave");
                        $(selector).find("a").off("mouseenter mouseleave");
                    });
                });
            </script>';
        }
    }
    
    /**
     * جلوگیری از چک آپدیت از API
     */
    public static function disable_update_checks($args, $url) {
        // قطع ارتباط با api.wordpress.org
        if (strpos($url, 'api.wordpress.org/plugins/update-check') !== false) {
            return $args;
        }
        
        // قطع ارتباط با api.wordpress.org/plugins/info
        if (strpos($url, 'api.wordpress.org/plugins/info') !== false) {
            return $args;
        }
        
        return $args;
    }

    /**
     * Short-circuit update requests با یک پاسخ خنثی
     */
    public static function short_circuit_update_requests($pre, $args, $url) {
        if (strpos($url, 'api.wordpress.org/plugins/update-check') !== false || strpos($url, 'api.wordpress.org/plugins/info') !== false) {
            return [
                'headers'  => [],
                'body'     => null,
                'response' => ['code' => 200, 'message' => 'OK'],
                'cookies'  => [],
            ];
        }
        return $pre;
    }
    
    /**
     * غیرفعال کردن Plugin API
     */
    public static function disable_plugin_api($false, $action, $args) {
        foreach (self::$gateway_plugins as $gateway) {
            $plugin_file = self::get_plugin_file($gateway);
            if ($plugin_file && isset($args->slug) && strpos($plugin_file, $args->slug) !== false) {
                return false;
            }
        }
        
        return $false;
    }
}

// راه‌اندازی کلاس
add_action('plugins_loaded', [IranDargah_Gateway_Cleanup::class, 'init'], 1);

